<?php
if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly.
}

class Ultimate_AI_User_Manager
{
  private $user_meta_key = "ultimate_ai_user_meta";
  private $block_meta_key = "ultimate_ai_is_user_block";
  private $user_word_token_meta_key = "ultimate_ai_user_word_token_amount";
  private $user_img_token_meta_key = "ultimate_ai_user_img_token_amount";
  private $user_tts_token_meta_key = "ultimate_ai_user_tts_token_amount";
  private $theme_settings_manager;
  public $TTS_VERSION_META_KEY = 'UAI_TTS_VERSION_META_KEY';
  public $TTS_VERSION = 2.5;

  public function deleteUser($user_id)
  {
    return wp_delete_user($user_id);
  }

  public function is_new_version_available()
  {
    $pre_ver = get_option($this->TTS_VERSION_META_KEY, false);

    if ($pre_ver == false || $this->TTS_VERSION > $pre_ver) {
      update_option($this->TTS_VERSION_META_KEY, $this->TTS_VERSION);
      return true;
    }
    return false;
  }

  public function migrate_user_meta_to_new_structure()
  {
    $users = $this->get_users(-1, 0);

    foreach ($users as $user => $data) {
      if (!isset($data['meta']['limits']['tts'])) {
        $data['meta']['limits']['tts'] = 0;
      }
      if (!isset($data['meta']['tts_token_amount'])) {
        $data['meta']['tts_token_amount'] = 0;
      }
      if (!isset($data['meta']['used_tts_token'])) {
        $data['meta']['used_tts_token'] = 0;
      }
      $updated = update_user_meta($data['user']->ID, $this->user_meta_key, $data);
    }
  }

  public function __construct()
  {
    $this->theme_settings_manager = new Ultimate_AI_Theme_Setting_Manager();
  }

  public function updateUserStatus($status, $order_id)
  {
    $order_meta = get_post_meta($order_id, 'ultimate_ai_order_meta', true);

    $user = get_user_by('ID', $order_meta['user_id']);


    $user_meta = get_user_meta($user->ID, 'ultimate_ai_user_meta', true);
    $user_meta['subscription']['payment_status'] = $status;


    if ($status == 'hold' || $status == 'ban' || $status == 'cancel') {
      $user_meta['subscription']['subscribed'] = false;
    }


    $updated = $this->update_user_meta($user, [
      'limits'        => $user_meta['limits'],
      'tools'         => $user_meta['tools'],
      'subscription'  => $user_meta['subscription'],
      'user_active'   => $user_meta['subscription']['subscribed']
    ]);

    return $updated;
  }

  public function add_new_user($user)
  {
    $user_id = wp_insert_user($user);

    if (is_wp_error($user_id)) {
      return [
        'status' => 401,
        'msgStatus' => 'error',
        'message' => esc_html__('User creation failed', ULTIMATE_AI_SLUG),
        'user_id' => null
      ];
    } else {
      wp_new_user_notification($user_id, null, 'both');
      return [
        'status' => 200,
        'msgStatus' => 'success',
        'message' => esc_html__('User creation successful', ULTIMATE_AI_SLUG),
        'user_id' => $user_id
      ];
    }

    return $user_id;
  }

  /**
   * 
   * This will add an user to a plan or a team.
   * Message sent via email.
   * When the user already exists it will be added to a plan or team directly
   * When user doesnt exists it will create an user with activation url sent to mail.
   *  
   */
  public function add_subscriber($subscriber_data)
  {
    $email = $subscriber_data['user_email'];
    $subscription = new Ultimate_AI_Subscription();

    $pricing_id = $subscription->getPricingID($subscriber_data['plan_id'], 0);

    if (email_exists($email)) {
      // if(subscribed){
      // 	// If already subscribed check it and do it accordingly
      // }
      // Get user id and add plan
      $user = get_user_by('email', $email);
      $user_id = $user->ID;
      $added = $this->addUsertoPlan($user_id, $email, $pricing_id, $subscriber_data['plan_interval']);
      $this->send_mail(
        [
          'to' => $email,
          'subject' => esc_html__('You got a subscription pack!', ULTIMATE_AI_SLUG),
          'message' => esc_html__('Congratulation! you just received a subscription pack please login to your account to enjoy it.', ULTIMATE_AI_SLUG)
        ]
      );
      return $added;
    } else {
      $newuser = array(
        'user_pass' =>  wp_hash_password($email),
        'user_login' => $email,
        'user_email' => $email,
      );
      $user_id = wp_insert_user($newuser);
      if ($user_id && !is_wp_error($user_id)) {
        $code = sha1($user_id . time());
        $activation_link = add_query_arg(array('activation_code' => $code), site_url('activation'));

        add_user_meta($user_id, 'user_activation_key', $code);
        $added = $this->addUsertoPlan($user_id, $email, $pricing_id, $subscriber_data['plan_interval']);
        $this->send_mail(
          [
            'to' => $email,
            'subject' => esc_html__('You got a subscription pack!', ULTIMATE_AI_SLUG),
            'message' => 'Congratulation! here is your activation link ' . $activation_link
          ]
        );
        return $added;
      } else {
        return [
          'status' => 'FAIL',
          'data' => null
        ];
      }
    }
  }

  public function send_mail($params)
  {
    $sent = wp_mail(
      $params['to'],
      $params['subject'],
      $params['message'],
    );
    return $sent;
  }


  public function addUsertoPlan($user_id, $email, $pricing_id, $plan_interval)
  {
    $subscription = new Ultimate_AI_Subscription();
    $order = [
      'subscribed'    => true,
      'user_id'       => $user_id,
      'name'          => $email,
      'email'         => $email,
      'customer_id'   => $user_id,
      'expiry'        => time() + 30 * 24 * 60 * 60,
      'payment_method_id' => 0,
      'order_id'       => rand(),
      'payment_method' => 'Self',
      'plan_price'    => 0,
      'plan_interval' => $plan_interval,
      'pricing_id'    => $pricing_id,
      'payment_status' => 'active',
      'starts'        => time(),
      'country'        => '',
      'address'        => '',
      'currency'      => $subscription->getDefaultCurrency(),
    ];
    return $subscription->subscribe($order);
  }

  public static function login_user($user_name, $password)
  {
    $user = wp_signon(
      array(
        "user_login"    => $user_name,
        "user_password" => $password,
        "remember"    => true,
      )
    );

    //check if success
    if (is_wp_error($user)) return false;

    wp_set_current_user($user->ID);
    wp_set_auth_cookie($user->ID);

    return $user;
  }

  public function login_user_passowrd_less($user)
  {
    wp_clear_auth_cookie();
    wp_set_current_user($user->ID);
    wp_set_auth_cookie($user->ID, true);
  }

  public static function logout_user()
  {
    wp_logout();
    return true;
  }

  public function get_full_user($user)
  {
    if (empty($user->ID)) return []; //no user pass send balnk array

    $meta = get_user_meta($user->ID, $this->user_meta_key, true);
    if (empty($meta)) $meta = [];

    //set total tools count and used tool  count
    $templates = (new Ultimate_AI_Template_Manager())->ultimate_ai_get_templates();
    $meta["tools_count"] = $templates["total_templates"] + 2; // 2 different tool,(chat,image). code is inside template so dont add it here 
    $meta["tools_used_count"] = 0;
    $meta["email_verified"] = get_user_meta($user->ID, 'uai_is_email_verified', true);

    if (!empty($meta["tool_uses"])) {
      $template_keys = array_map(function ($item) {
        return $item["key"];
      }, $templates["templates"]);
      $template_keys[] = "default_chat_tool";
      $template_keys[] = "generate_image";

      foreach ($template_keys as $_k) {
        if (!empty($meta["tool_uses"][$_k])) {
          $meta["tools_used_count"] += 1;
        }
      }
    }



    //check if essencial key exist, if not then set them
    $key_img_t_a = "image_token_amount";
    $key_word_t_a = "word_token_amount";
    $key_tts_t_a = "tts_token_amount";
    $key_tools = "tools";
    $key_limits = "limits";

    if (!array_key_exists($key_img_t_a, $meta)) $meta[$key_img_t_a] = 0;
    if (!array_key_exists($key_word_t_a, $meta)) $meta[$key_word_t_a] = 0;
    if (!array_key_exists($key_tts_t_a, $meta)) $meta[$key_tts_t_a] = 0;
    if (!array_key_exists($key_limits, $meta)) {
      $meta[$key_limits] = ["words" => 0, "images" => 0, "tts" => 0, "users" => 1];
    }
    if (!array_key_exists($key_tools, $meta)) {
      $meta[$key_tools] = ["advanced_tools" => [], "tools" => []];
    } else {
      $pricing_id = $meta['subscription']['pricing_id'] ?? false;
      if ($pricing_id) {
        $subscription = new Ultimate_AI_Subscription();
        $tools = $subscription->getPackageTools($pricing_id);
        $meta['tools'] = $tools;
      }
    }

    $_user = get_userdata($user->ID);
    //remove user password
    if (isset($_user->user_pass)) {
      $_user->user_pass = "";
    }

    if (isset($_user->roles)) {
      $_user->roles = array_values($_user->roles);
    }
    return array("user" => $_user, "meta" => $meta);
  }



  /**
   * Get All users from the database
   * accepts : limit per results, offset
   */
  public function get_users($limit, $skip_result)
  {
    $users = get_users(
      array(
        "number" => (int)$limit,
        "offset" => (int)$skip_result,
      )
    );

    $full_users = array();

    foreach ($users as $user) {
      $full_users[] =  $this->get_full_user($user);
    }

    return $full_users;
  }

  public function register_user($user_name, $password, $email = "", $is_3rd_party_login = false)
  {
    $user_exist = Ultimate_AI_User_Manager::user_exist($user_name);
    if ($user_exist) return false;

    $user_id = wp_create_user($user_name, $password, $email);
    if (is_wp_error($user_id)) return false;
    update_user_meta($user_id, "is_3rd_party_login", $is_3rd_party_login);

    // FOR EMAIL VERIFICATION PURPOSE
    $app_settings = $this->theme_settings_manager->ultimate_ai_get_theme_settings(['uai_app_settings']);
    $verification_req = isset($app_settings['uai_app_settings']['email_verification_required']) ? $app_settings['uai_app_settings']['email_verification_required'] : 'false';

    if ($verification_req == 'true') {
      $this->send_verification_mail($user_id, $email);
    }

    //user created. Now log the user in system
    $user = Ultimate_AI_User_Manager::login_user($user_name, $password);
    return $user;
  }

  public function send_verification_mail($user_id, $email)
  {
    $verification_key = uniqid() . rand();
    update_user_meta($user_id, "uai_user_verification_key", $verification_key);
    update_user_meta($user_id, "uai_is_email_verified", 'false');

    $dashboard_page_id = get_option('ultimate_ai_dashboard_page_id');
    $varification_link = add_query_arg(array('verification_key' => $verification_key), get_the_permalink($dashboard_page_id));

    $this->send_mail(
      [
        'to' => $email,
        'subject' => esc_html__('Please verify your email!', ULTIMATE_AI_SLUG),
        'message' => esc_html__('Congratulation! your registration is complete at ' . get_bloginfo('name') . '. Please verify your account by following this link: ' . $varification_link, ULTIMATE_AI_SLUG)
      ]
    );
  }

  public static function user_exist($user_name)
  {
    return username_exists($user_name);
  }

  public static function get_user_by($field, $value)
  {
    $user = get_user_by($field, $value);
    return $user ? $user : false;
  }

  public function block_user($user)
  {
    $is_success = update_user_meta($user->ID, $this->block_meta_key, true);
    $this->update_user_meta($user, array("is_blocked" => true));
    return $is_success ? true : false;
  }

  public function unblock_user($user)
  {
    $is_success = update_user_meta($user->ID, $this->block_meta_key, false);
    $this->update_user_meta($user, array("is_blocked" => false));
    return $is_success ? true : false;
  }

  public function get_user_token($user, $token_type)
  { //$token_type = word/image
    if ($token_type == "word") {
      $token = get_user_meta($user->ID, $this->user_word_token_meta_key, true);
    }
    if ($token_type == "image") {
      $token = get_user_meta($user->ID, $this->user_img_token_meta_key, true);
    }

    if (empty($token)) return 0;
    return $token;
  }

  //$token_type = word/image , $operation_type = "INCREMENT"/"DECREMENT",
  //$do_reset_token = true/false. Will reset the prev token to 0 if 'true'
  public function update_user_token($user, $new_token_amount, $token_type, $operation_type, $do_reset_token = false)
  {
    $m_key = $token_type == "word" ? $this->user_word_token_meta_key : ($token_type == "image" ? $this->user_img_token_meta_key : ($token_type == "tts" ? $this->user_tts_token_meta_key : ""));
    $n_key = $token_type == "word" ? "word_token_amount" : ($token_type == "image" ? "image_token_amount" : ($token_type == "tts" ? "tts_token_amount" : ""));
    //"word_token_amount"/"image_token_amount" is left image and word token amount

    $is_success = false;

    if ($m_key && $n_key) {
      $token = get_user_meta($user->ID, $m_key, true);
      if (empty($token) || $do_reset_token) $token = 0;

      if ($this->is_user_unlimited($token_type, $user)) {
        $token = -1;
      } else {
        if ($operation_type == "INCREMENT") $token = (int)$token + $new_token_amount;
        if ($operation_type == "DECREMENT") $token = $token <= $new_token_amount ? 0 : $token - $new_token_amount;

        $is_success = update_user_meta($user->ID, $m_key, $token);
        $this->update_user_meta($user, array($n_key => $token));
      }

      //KEEP TRACK OF USER USED TOKEN AMOUNT
      //if $operation_type == "DECREMENT" then user used new token
      if ($operation_type == "DECREMENT") {
        $full_user = $this->get_full_user($user);
        $_key = $token_type == "word" ? "used_word_token" : ($token_type == "image" ? "used_image_token" : ($token_type == "tts" ? "used_tts_token" : ""));
        if ($_key && !empty($full_user["meta"][$_key])) {
          //increment from previous value
          $_value = (int) $full_user["meta"][$_key] + $new_token_amount;
          $this->update_user_meta($user, array($_key => $_value));
        } else {
          //add new value
          $this->update_user_meta($user, array($_key => $new_token_amount));
        }
      }
    }

    return $is_success;
  }

  public function is_user_subscriber($user)
  {
    $meta = get_user_meta($user->ID, $this->user_meta_key, true);
    if (!isset($meta['subscription']['subscribed']) || empty($meta['subscription']['subscribed'])) return false;
    if ($meta['subscription']['subscribed']) {
      return true;
    } else {
      return false;
    }
  }


  public function get_subscription_pricing_id($user)
  {
    $meta = get_user_meta($user->ID, $this->user_meta_key, true);
    if (!isset($meta['subscription']['pricing_id']) || empty($meta['subscription']['pricing_id'])) return false;
    if ($meta['subscription']['pricing_id']) {
      return $meta['subscription']['pricing_id'];
    } else {
      return false;
    }
  }

  // $keys = open_ai / 
  public function get_user_integration_settings($user, $key)
  {
    $meta = get_user_meta($user->ID, $this->user_meta_key, true);
    if (!isset($meta[$key]) || empty($meta[$key])) return false;
    if ((array) $meta[$key] != NULL) {
      return (array) $meta[$key];
    } else {
      return false;
    }
  }

  public function check_user_integration_access($key)
  {
    $subscription = new Ultimate_AI_Subscription();
    if (!function_exists('wp_get_current_user')) {
      include(ABSPATH . "wp-includes/pluggable.php");
    }
    $user = wp_get_current_user();

    $roles = $user->roles;

    $roles_to_check = ["administrator", "uai_dashboard_viewer"];

    $roles_has_access = array_intersect($roles, $roles_to_check);

    $settings = '';
    $has_user_access = false;

    // CHECK CURRENT SUBSCRIPTION PLAN CAN SET API KEY.
    $pricing_id = $this->get_subscription_pricing_id($user);
    if ($pricing_id) {
      $subscription_has_integration = $subscription->getPackageIntegrationAccess($pricing_id, $key);
      if ($subscription_has_integration) {
        $has_user_access = true;
        $settings = $this->get_user_integration_settings($user, $key);
      }
    } else {
      // CHECK IF FREE USER CAN SET OPEN AI KEY
      $app_settings = $this->theme_settings_manager->ultimate_ai_get_theme_settings(["uai_app_settings"]);
      $free_integration = $app_settings["uai_app_settings"]["integration_" . $key] ?? false;

      if ($free_integration == true || $free_integration == 'true') {
        $has_user_access = true;
        $settings = $this->get_user_integration_settings($user, $key);
      }
    }

    if (count($roles_has_access) > 0) {
      $has_user_access = true;
      $settings = $this->get_user_integration_settings($user, $key);
    }

    return [
      'settings' => $settings,
      'has_access' => $has_user_access,
    ];
  }

  public function getUserBySubscriptionID($subscription_id)
  {
    $args = array(
      'fields' => 'ID',
      'meta_query' => array(
        array(
          'key'     => $this->user_meta_key,
          'value'   => sprintf(':"%s";', $subscription_id),
          'compare' => 'LIKE'
        )
      )
    );
    $user_query = get_users($args);
    return $user_query[0];
  }

  public function update_user_meta($user, $new_meta)
  {
    //$new_meta structure: array( "key" => value )
    $meta = get_user_meta($user->ID, $this->user_meta_key, true);
    if (empty($meta)) $meta = array();

    foreach ($new_meta as $key => $value) {
      $meta[$key] = $value;
    }

    $is_success = update_user_meta($user->ID, $this->user_meta_key, $meta);

    return $is_success ? true : false;
  }

  public function filter_users($args = [])
  {
    $user_args = [
      'number' => -1,
      'offset' => 0,
      'fields' => ['ID', 'registered'],
    ];

    $new_args = array_merge($user_args, $args);

    $user_query = new WP_User_Query($new_args);

    $users = [];

    foreach ($user_query->get_results() as $key => $user) {

      $meta = get_user_meta($user->ID, 'ultimate_ai_user_meta', true);
      $full_user = $this->get_full_user($user);
      $temp = [
        'meta' => $meta,
        'words' => isset($meta['limits']['words']) ? $meta['limits']['words'] == -1 ? 'Unlimited' : $meta['limits']['words'] : 0,
        'images' => isset($meta['limits']['images']) ? $meta['limits']['images'] == -1 ? 'Unlimited' : $meta['limits']['images'] : 0,
        'tts' => isset($meta['limits']['tts']) ? $meta['limits']['tts'] == -1 ? 'Unlimited' : $meta['limits']['tts'] : 0,
        'country' => get_user_meta($user->ID, 'country', true),
        'full_user' => $full_user,
      ];

      $new_user = array_merge($temp, (array) $user);
      $users[$key] = $new_user;
    }

    return [
      'users_count' => $user_query->get_total(),
      'users' => $users,
    ];
  }

  public function sendPasswordresetLink($user)
  {
    $results = retrieve_password($user->user_login);

    if (true === $results) {
      return [
        'success' => true,
        'message' => esc_html__('A password reset link is sent to the mail', ULTIMATE_AI_SLUG),
      ];
    } else {
      return [
        'success' => false,
        'message' => $results->get_error_message(),
      ];
    }
  }

  public function update_user_role(string $role, int $user_id)
  {
    $supported_roles = ["administrator", "subscriber"];
    if (!in_array($role, $supported_roles)) {
      return ["status" => false, "message" => "Unsupported role!"];
    }

    $user = new WP_User($user_id);
    $user->set_role($role);
    return ["status" => true, "message" => "Role updated!"];
  }

  public function is_user_unlimited($token_type = 'word', $__user = false)
  {

    $app_settings = $this->theme_settings_manager->ultimate_ai_get_theme_settings(['uai_app_settings']);
    $unmetered = $app_settings['uai_app_settings']['unmetered_usage'] ?? false;

    if ($unmetered == true || $unmetered == 'true') {
      return true;
    }

    $_user  = $__user ? $__user : wp_get_current_user();

    if ($_user->ID == 0) return false;

    $user = $this->get_full_user($_user);
    //if administrator then has unlimited accessh
    if (in_array("administrator", $user["user"]->roles)) return true;

    if ($token_type == 'tts') {
      $limit = $user["meta"]["limits"][$token_type];
    } else {
      $limit = $user["meta"]["limits"][$token_type . 's'];
    }

    if ($limit == -1) {
      return true;
    }

    return false;
  }

  public function change_password($user, $new_pass)
  {
    wp_set_password($new_pass, $user->ID);
    $this->logout_user();
    return $this->login_user($user->user_login, $new_pass);
  }
}


//new Ultimate_AI_User_Manager();
